#include "handleCache.hpp"

#include "storage.hpp"

#include <gflags/gflags.h>
#include <monitoring/monitoring.h>

DEFINE_uint64(max_open_descriptors, 1000,
    "maximum number of unused file descriptors open at one time (per storage)");
DEFINE_uint64(descriptor_cache_duration, 60000,
    "maximum duration after which a unused descriptor will be removed from the "
    "cache");
DEFINE_uint64(descriptor_cache_tick, 1000,
    "cache will be pruned every descriptor_cache_tick "
    "milliseconds");

namespace rtransfer {

void HandleCache::trigger()
{
    executor_.add(std::bind(&HandleCache::doTrigger, this));
}

void HandleCache::doTrigger()
{
    auto threshold = std::chrono::steady_clock::now() -
                     std::chrono::milliseconds{FLAGS_descriptor_cache_duration};

    auto it = lru_.rbegin();
    std::size_t size = lru_.size();
    std::size_t totalErased = 0;

    while (it != lru_.rend() && (size > FLAGS_max_open_descriptors ||
                                    it->addTimePoint < threshold)) {
        handles_.erase(it->key);
        --size;
        ++totalErased;
        ++it;
    }

    ONE_METRIC_COUNTER_SUB("handle_cache.handles.concurrent", totalErased);
    lru_.erase(it.base(), lru_.end());
}

HandleCache::HandleCache()
    : handles_{FLAGS_max_open_descriptors}
    , periodicHandler_{std::chrono::milliseconds{FLAGS_descriptor_cache_tick},
          std::bind(&HandleCache::trigger, this)}
{
}

folly::Future<folly::IOBufQueue> HandleCache::read(Storage &storage,
    std::uint64_t reqId, folly::fbstring fileId, folly::fbstring fileGuid,
    const std::uint64_t offset, const std::size_t size, std::uint8_t priority)
{
    return withHandle(storage, fileId, fileGuid, OpenType::read,
        [=, &storage](one::helpers::FileHandlePtr handle) {
            VLOG(2) << "Reading " << reqId;
            return storage.read(handle, reqId, offset, size, priority);
        });
}

folly::Future<folly::Unit> HandleCache::write(Storage &storage,
    std::uint64_t reqId, folly::fbstring fileId, folly::fbstring fileGuid,
    const std::uint64_t offset, folly::IOBufQueue buf, std::uint8_t priority)
{
    return withHandle(storage, fileId, fileGuid, OpenType::write,
        [=, buf = std::move(buf), &storage](
            one::helpers::FileHandlePtr handle) mutable {
            return storage.write(
                handle, reqId, offset, std::move(buf), priority);
        });
}

folly::Future<folly::Unit> HandleCache::sync(
    Storage &storage, folly::fbstring fileId, folly::fbstring fileGuid)
{
    return withHandle(storage, fileId, fileGuid, OpenType::write,
        [=, &storage](one::helpers::FileHandlePtr handle) mutable {
            return handle->fsync(true);
        });
}

template <typename Cb>
auto HandleCache::withHandle(Storage &storage, folly::fbstring fileId,
    folly::fbstring fileGuid, OpenType openType, Cb &&cb)
    -> decltype(cb(std::declval<one::helpers::FileHandlePtr>()))
{
    return via(&executor_)
        .then([=, &storage] {
            return checkout(storage, fileId, fileGuid, openType);
        })
        .via(&executor_)
        .then(std::forward<Cb>(cb))
        .via(&executor_)
        .ensure([=, &storage] { checkin(fileGuid, openType); });
}

folly::Future<one::helpers::FileHandlePtr> HandleCache::checkout(
    Storage &storage, folly::fbstring fileId, folly::fbstring fileGuid,
    OpenType openType)
{
    ONE_METRIC_COUNTER_INC("handle_cache.checkouts.concurrent");
    ONE_METRIC_COUNTER_INC("handle_cache.checkouts.total");

    bool inserted;
    decltype(handles_)::iterator it;
    std::tie(it, inserted) =
        handles_.emplace(std::make_pair(fileGuid, openType), Node{});

    auto &node = it->second;
    ++node.refCount;

    if (!inserted) {
        if (node.lruIt != lru_.end()) {
            lru_.erase(node.lruIt);
            node.lruIt = lru_.end();
        }
        auto handleFuture = node.promise->getFuture();
        if (!handleFuture.isReady() || !handleFuture.hasException())
            return handleFuture;
    }

    node.lruIt = lru_.end();
    node.promise =
        std::make_shared<folly::SharedPromise<one::helpers::FileHandlePtr>>();

    ONE_METRIC_COUNTER_INC("handle_cache.handles.concurrent");
    ONE_METRIC_COUNTER_INC("handle_cache.handles.total");

    storage.open(std::move(fileId), openType)
        .then([promise = node.promise](
                  folly::Try<one::helpers::FileHandlePtr> handle) {
            promise->setTry(std::move(handle));
        });

    return node.promise->getFuture();
}

void HandleCache::checkin(folly::fbstring fileGuid, OpenType openType)
{
    ONE_METRIC_COUNTER_INC("handle_cache.checkouts.total");

    auto key = std::make_pair(fileGuid, openType);
    auto &node = handles_.at(key);
    if (--node.refCount == 0) {
        auto now = std::chrono::steady_clock::now();
        node.lruIt = lru_.emplace(lru_.begin(), LRUNode{std::move(key), now});
    }
}

}  // namespace rtransfer
