#pragma once

#include "openType.hpp"
#include "periodicHandler.hpp"

#include <folly/EvictingCacheMap.h>
#include <folly/FBString.h>
#include <folly/Hash.h>
#include <folly/Optional.h>
#include <folly/executors/SerialExecutor.h>
#include <folly/futures/SharedPromise.h>
#include <helpers/storageHelper.h>

#include <list>
#include <unordered_map>

namespace rtransfer {

class Storage;

class HandleCache {
public:
    HandleCache();

    folly::Future<folly::IOBufQueue> read(Storage &storage, std::uint64_t reqId,
        folly::fbstring fileId, folly::fbstring fileGuid,
        const std::uint64_t offset, const std::size_t size,
        std::uint8_t priority);

    folly::Future<folly::Unit> write(Storage &storage, std::uint64_t reqId,
        folly::fbstring fileId, folly::fbstring fileGuid,
        const std::uint64_t offset, folly::IOBufQueue buf,
        std::uint8_t priority);

    folly::Future<folly::Unit> sync(
        Storage &storage, folly::fbstring fileId, folly::fbstring fileGuid);

private:
    using Key = std::pair<folly::fbstring, OpenType>;

    struct LRUNode {
        Key key;
        std::chrono::steady_clock::time_point addTimePoint;
    };

    struct Node {
        std::shared_ptr<folly::SharedPromise<one::helpers::FileHandlePtr>>
            promise;
        std::list<LRUNode>::iterator lruIt;
        std::size_t refCount = 0;
    };

    template <typename Cb>
    auto withHandle(Storage &storage, folly::fbstring fileId,
        folly::fbstring fileGuid, OpenType openType, Cb &&cb)
        -> decltype(cb(std::declval<one::helpers::FileHandlePtr>()));

    folly::Future<one::helpers::FileHandlePtr> checkout(Storage &storage,
        folly::fbstring fileId, folly::fbstring fileGuid, OpenType openType);

    void checkin(folly::fbstring fileGuid, OpenType openType);

    void trigger();
    void doTrigger();

    folly::SerialExecutor executor_;
    std::unordered_map<Key, Node> handles_;
    std::list<LRUNode> lru_;

    PeriodicHandler periodicHandler_;
};

}  // namespace rtransfer
