#include "storage.hpp"

#include "uuid.hpp"

#include <folly/FBString.h>
#include <folly/Function.h>
#include <folly/String.h>
#include <folly/executors/NamedThreadFactory.h>
#include <folly/executors/ThreadPoolExecutor.h>
#include <folly/io/IOBufQueue.h>
#include <gflags/gflags.h>
#include <helpers/storageHelper.h>
#include <monitoring/monitoring.h>

#include <atomic>
#include <chrono>
#include <queue>
#include <unordered_map>

DEFINE_uint64(
    storage_buckets, 100, "max number of in-progress operations on storage");

namespace {
folly::Future<folly::IOBufQueue> readLoop(rtransfer::Storage::Bucket &bucket,
    one::helpers::FileHandlePtr handle, std::uint64_t offset, std::size_t size,
    folly::IOBufQueue buf)
{
    if (size == 0)
        return folly::makeFuture<folly::IOBufQueue>(std::move(buf));

    ONE_METRIC_COUNTER_ADD("link.storage.read.in_progress", 1);
    ONE_METRIC_COUNTER_ADD("link.storage.read.in_progress_bytes", size);

    auto readFuture =
        handle->isConcurrencyEnabled()
            ? handle->read(offset, size)
            : via(&bucket.executor, [=] { return handle->read(offset, size); });

    return readFuture.via(folly::getCPUExecutor().get())
        .then([=, &bucket, buf = std::move(buf)](
                  folly::IOBufQueue nextBuf) mutable {
            ONE_METRIC_COUNTER_SUB("link.storage.read.in_progress", 1);
            ONE_METRIC_COUNTER_SUB("link.storage.read.in_progress_bytes", size);
            if (nextBuf.empty()) {
                size = 0;
            }
            else {
                assert(size >= nextBuf.chainLength());
                size -= nextBuf.chainLength();
                offset += nextBuf.chainLength();
                buf.append(std::move(nextBuf));
            }
            return readLoop(
                bucket, std::move(handle), offset, size, std::move(buf));
        });
}

folly::Future<folly::Unit> writeLoop(rtransfer::Storage::Bucket &bucket,
    one::helpers::FileHandlePtr handle, std::uint64_t offset,
    folly::IOBufQueue buf)
{
    if (buf.empty())
        return folly::makeFuture();

    folly::IOBufQueue cloneBuf{folly::IOBufQueue::cacheChainLength()};
    cloneBuf.append(buf.front()->clone());

    ONE_METRIC_COUNTER_ADD("link.storage.write.in_progress", 1);
    ONE_METRIC_COUNTER_ADD(
        "link.storage.write.in_progress_bytes", cloneBuf.chainLength());

    auto writeFuture =
        handle->isConcurrencyEnabled()
            ? handle->write(offset, std::move(buf))
            : via(&bucket.executor, [=, buf = std::move(buf)]() mutable {
                  return handle->write(offset, std::move(buf));
              });

    return writeFuture.via(folly::getCPUExecutor().get())
        .then([=, &bucket, cloneBuf = std::move(cloneBuf)](
                  std::size_t wrote) mutable {
            ONE_METRIC_COUNTER_SUB("link.storage.write.in_progress", 1);
            ONE_METRIC_COUNTER_SUB(
                "link.storage.write.in_progress_bytes", cloneBuf.chainLength());

            cloneBuf.trimStart(wrote);
            return writeLoop(
                bucket, std::move(handle), offset, std::move(cloneBuf));
        });
}

template <typename Queue, typename Cb>
void usingBucket(Queue &queue, rtransfer::Storage::Bucket &bucket, Cb &&cb)
{
    folly::getCPUExecutor()->add(
        [=, &queue, &bucket, cb = std::forward<Cb>(cb)]() mutable {
            while (true) {
                decltype(queue.impl.pop()) reqs;
                {
                    std::lock_guard<std::mutex> guard{queue.mutex};
                    reqs = queue.impl.pop();
                    if (reqs.empty()) {
                        queue.isUsed = false;
                        return;
                    }
                }
                cb(std::move(reqs));
            }
        });
}

template <typename T>
void logRequests(const char *what, const folly::fbvector<T> &reqs)
{
    if (!VLOG_IS_ON(2))
        return;

    std::stringstream log;
    for (auto &req : reqs)
        log << " {reqId: " << req.reqId << ", offset: " << req.offset
            << ", size: " << req.getSize() << "}";
    VLOG(2) << what << " requests:" << log.str();
}

void handleRead(rtransfer::Storage::Bucket &bucket)
{
    usingBucket(bucket.readQueue, bucket,
        [&bucket](folly::fbvector<rtransfer::Storage::ReadRequest> reqs) {
            logRequests("Reading", reqs);
            auto handle = reqs.front().handle;
            auto offset = reqs.front().offset;
            std::size_t size = 0;
            for (auto &req : reqs)
                size += req.size;

            readLoop(bucket, handle, offset, size,
                folly::IOBufQueue{folly::IOBufQueue::cacheChainLength()})
                .then([size, reqs = std::move(reqs)](
                          folly::Try<folly::IOBufQueue> buft) mutable {
                    const std::string m{"link.storage.read.queue"};
                    ONE_METRIC_COUNTER_SUB(m + "_bytes", size);
                    ONE_METRIC_COUNTER_SUB(m, 1);

                    if (buft.hasValue()) {
                        logRequests("Read", reqs);
                        auto buf = std::move(buft.value());
                        for (auto &req : reqs) {
                            folly::IOBufQueue subQueue{
                                folly::IOBufQueue::cacheChainLength()};
                            subQueue.append(buf.split(req.size));
                            req.promise.setValue(std::move(subQueue));
                        }
                    }
                    else {
                        logRequests("Failed to read", reqs);
                        for (auto &req : reqs)
                            req.promise.setException(buft.exception());
                    }
                });
        });
}

void handleWrite(rtransfer::Storage::Bucket &bucket)
{
    usingBucket(bucket.writeQueue, bucket,
        [&bucket](folly::fbvector<rtransfer::Storage::WriteRequest> reqs) {
            logRequests("Writing", reqs);
            auto handle = reqs.front().handle;
            auto offset = reqs.front().offset;
            folly::IOBufQueue buf{folly::IOBufQueue::cacheChainLength()};
            for (auto &req : reqs)
                buf.append(std::move(req.buf));

            const std::size_t bufSize = buf.chainLength();

            writeLoop(bucket, handle, offset, std::move(buf))
                .then([bufSize, reqs = std::move(reqs)](
                          folly::Try<folly::Unit> t) mutable {
                    const std::string m{"link.storage.write.queue"};
                    ONE_METRIC_COUNTER_SUB(m + "_bytes", bufSize);
                    ONE_METRIC_COUNTER_SUB(m, 1);

                    if (t.hasValue()) {
                        logRequests("Wrote", reqs);
                        for (auto &req : reqs)
                            req.promise.setValue();
                    }
                    else {
                        logRequests("Failed to write", reqs);
                        for (auto &req : reqs)
                            req.promise.setException(t.exception());
                    }
                });
        });
}

}  // namespace

namespace rtransfer {

Storage::Storage(
    folly::fbstring storageId, one::helpers::StorageHelperPtr storageHelper)
    : storageIdHex_{folly::hexlify(storageId)}
    , storageHelper_{std::move(storageHelper)}
{
    LOG(INFO) << "Adding storage " << storageIdHex_;
    for (std::size_t i = 0; i < FLAGS_storage_buckets; ++i)
        buckets_.emplace_back(
            std::make_unique<Bucket>(folly::getCPUExecutor()));
}

folly::Future<one::helpers::FileHandlePtr> Storage::open(
    folly::fbstring fileId, OpenType openType)
{
    VLOG(2) << "Opening file '" << fileId << "'";
    const int openFlag = openType == OpenType::read ? O_RDONLY : O_WRONLY;
    return storageHelper_->open(fileId, openFlag, {});
}

template <typename Req, typename Cb>
auto Storage::inBucket(Req &&req, Cb &&cb)
{
    auto future = req.promise.getFuture();
    std::size_t bucketNum =
        std::hash<void *>{}(static_cast<void *>(req.handle.get())) %
        buckets_.size();
    std::forward<Cb>(cb)(*buckets_[bucketNum], std::forward<Req>(req));
    return future;
}

template <typename Queue, typename Req, typename F>
void pushQueue(
    rtransfer::Storage::Bucket &bucket, Queue &queue, Req &&req, F &&handleFun)
{
    bool shouldIHandle = false;
    {
        std::lock_guard<std::mutex> guard{queue.mutex};
        queue.impl.push(std::forward<Req>(req));
        shouldIHandle = !queue.isUsed;
        queue.isUsed = true;
    }
    if (shouldIHandle)
        std::forward<F>(handleFun)(bucket);
}

folly::Future<folly::IOBufQueue> Storage::read(
    one::helpers::FileHandlePtr handle, std::uint64_t reqId,
    const std::uint64_t offset, const std::size_t size,
    const std::uint8_t priority)
{
    const std::string m{"link.storage.read.queue"};
    ONE_METRIC_COUNTER_ADD(m + "_bytes", size);
    ONE_METRIC_COUNTER_ADD(m, 1);

    return inBucket(ReadRequest{reqId, {}, std::move(handle), offset, size},
        [this, priority, reqId](Bucket &bucket, ReadRequest &&req) {
            VLOG(2) << "Pushing " << reqId << " into readQueue bucket "
                    << static_cast<void *>(&bucket);
            pushQueue(bucket, bucket.readQueue, std::move(req), handleRead);
        });
}

folly::Future<folly::Unit> Storage::write(one::helpers::FileHandlePtr handle,
    std::uint64_t reqId, const std::uint64_t offset, folly::IOBufQueue buf,
    const std::uint8_t priority)
{
    const std::string m{"link.storage.write.queue"};
    ONE_METRIC_COUNTER_ADD(m + "_bytes", buf.chainLength());
    ONE_METRIC_COUNTER_ADD(m, 1);

    return inBucket(
        WriteRequest{reqId, {}, std::move(handle), offset, std::move(buf)},
        [this, reqId](Bucket &bucket, WriteRequest &&req) {
            VLOG(2) << "Pushing " << reqId << " into writeQueue bucket "
                    << static_cast<void *>(&bucket);
            pushQueue(bucket, bucket.writeQueue, std::move(req), handleWrite);
        });
}

std::string Storage::m(const char *name)
{
    return "storage." + storageIdHex_.toStdString() + name;
}

}  // namespace rtransfer
