#pragma once

#include "common.hpp"
#include "proto/control.pb.h"
#include "serverSideLink.hpp"
#include "storage.hpp"

#include <folly/FBString.h>
#include <folly/FBVector.h>
#include <folly/ThreadName.h>
#include <folly/executors/IOThreadPoolExecutor.h>
#include <helpers/storageHelper.h>
#include <helpers/storageHelperCreator.h>
#include <asio/io_service.hpp>

#include <thread>

namespace rtransfer {

class StorageHelperFactory {
public:
    StorageHelperFactory(
        int threads, int webdavThreads, ClientLinksMap &clientLinks, ServerLinksMap &serverLinks)
        : helperIOService_{threads}
        , clientLinks_{clientLinks}
        , serverLinks_{serverLinks}
    {
        for (int i = 0; i < threads; ++i)
            helperWorkers_.emplace_back(std::thread{[this] {
                folly::setThreadName("helperIOS");
                helperIOService_.run();
            }});

        webDAVExecutor_ =
            std::make_shared<folly::IOThreadPoolExecutor>(webdavThreads);
    }

    ~StorageHelperFactory()
    {
        helperIOService_.stop();
        for (auto &thread : helperWorkers_)
            thread.join();

        webDAVExecutor_->stop();
    }

    std::shared_ptr<Storage> operator()(
        const link_control::proto::CreateHelper &req)
    {
        std::unordered_map<folly::fbstring, folly::fbstring> params;
        for (auto &param : req.params())
            params.emplace(
                folly::fbstring{param.key()}, folly::fbstring{param.value()});

        auto helper = helperCreator_.getStorageHelper(
            req.name(), params, req.io_buffered());

        return std::make_shared<Storage>(req.storage_id(), std::move(helper));
    }

private:
    asio::io_service helperIOService_;
    ClientLinksMap &clientLinks_;
    ServerLinksMap &serverLinks_;

    asio::io_service::work helperWork_{helperIOService_};
    folly::fbvector<std::thread> helperWorkers_;

    std::shared_ptr<folly::IOThreadPoolExecutor> webDAVExecutor_;

    one::helpers::StorageHelperCreator helperCreator_{helperIOService_,
        helperIOService_, helperIOService_, helperIOService_, helperIOService_,
        helperIOService_, webDAVExecutor_, helperIOService_};
};

}  // namespace rtransfer
