%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2017 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_request).
-author("Konrad Zemek").

%%%===================================================================
%%% Type definitions
%%%===================================================================

-type trimmed_notify_fun() ::
        fun((non_neg_integer(), pos_integer()) -> any()).

-type t() :: #{
         provider_id := binary(),
         file_guid := binary(),
         src_storage_id := binary(),
         src_file_id := binary(),
         dest_storage_id := binary(),
         dest_file_id := binary(),
         space_id := binary(),
         offset := non_neg_integer(),
         size := pos_integer(),
         priority := 0..255,

         ref => reference(),
         req_id => non_neg_integer(),
         conn => pid(),
         notify => rtransfer_link:notify_fun(),
         on_complete => rtransfer_link:on_complete_fun(),
         transfer_data => binary()
        }.

%%%===================================================================
%%% Exports
%%%===================================================================

-export([do_fetch/3, handle_fetch_updates/2]).
-export_type([t/0]).

%%%===================================================================
%%% API
%%%===================================================================

-spec do_fetch(ConnectionId :: binary(), Req :: t(), MonitorPid :: pid()) ->
    {ok, rtransfer_link_port:req_id(), reference(), rtransfer_link:notify_fun(),
        rtransfer_link:on_complete_fun()} | {error, term()}.
do_fetch(ConnectionId, #{on_complete := OnCompleteFun,
                         notify := NotifyFun0,
                         ref := Ref} = Req, MonitorPid) ->
    guard(Ref, OnCompleteFun, MonitorPid),
    NotifyFun = fun(O, S) -> NotifyFun0(Ref, O, S) end,
    try
        ReqId = do_fetch_helper(ConnectionId, Req),
        {ok, ReqId, Ref, NotifyFun, OnCompleteFun}
    catch
        Reason ->
            OnCompleteFun(Ref, {error, Reason}),
            {error, Reason}
    end.

-spec handle_fetch_updates(map(), trimmed_notify_fun()) -> term().
handle_fetch_updates(Resp, NotifyFun) ->
    ensure(connection, Resp),
    case Resp of
        #{<<"progress">> := Progress} ->
            Offset = to_integer(maps:get(<<"offset">>, Progress, 0)),
            Size = to_integer(maps:get(<<"size">>, Progress, 0)),
            NotifyFun(Offset, Size),
            Resp;
        Other ->
            Other
    end.

%%%===================================================================
%%% Helpers
%%%===================================================================

-spec do_fetch_helper(ConnectionId :: binary(),Req :: t()) ->
    rtransfer_link_port:req_id().
do_fetch_helper(ConnectionId,
         #{file_guid := FileGuid,
           src_storage_id := SrcStorageId,
           src_file_id := SrcFileId,
           dest_storage_id := DestStorageId,
           dest_file_id := DestFileId,
           offset := Offset,
           size := Size,
           priority := Priority,
           req_id := ReqId,
           transfer_data := TransferData}) ->
    FetchReq = #{fetch => #{connection_id => ConnectionId,
                            src_storage_id => base64:encode(SrcStorageId),
                            src_file_id => base64:encode(SrcFileId),
                            dest_storage_id => base64:encode(DestStorageId),
                            dest_file_id => base64:encode(DestFileId),
                            file_guid => base64:encode(FileGuid),
                            offset => Offset,
                            size => Size,
                            priority => Priority,
                            req_id => ReqId,
                            transfer_data => base64:encode(TransferData)}},

    {ok, _Handle} = rtransfer_link_callback:open(FileGuid, write),
    rtransfer_link_port:request(FetchReq).

-spec ensure(Type :: atom(), {error, term()} | Other) ->
                    Other | no_return()
                        when Other :: term().
ensure(Type, {error, Reason}) ->
    throw({Type, Reason});
ensure(_Type, Result) ->
    Result.

-spec to_integer(integer() | binary()) -> integer().
to_integer(Val) when is_integer(Val) -> Val;
to_integer(Val) when is_binary(Val) -> binary_to_integer(Val).

-spec guard(Ref :: reference(), rtransfer_link:on_complete_fun(),
    MonitorPid :: pid()) -> any().
guard(Ref, OnCompleteFun, MonitorPid) ->
    rtransfer_link_monitor:monitor(MonitorPid, Ref, OnCompleteFun).
